document.addEventListener("DOMContentLoaded", () => {
  const form = document.getElementById("formMovimiento");
  const tipoMovimiento = document.getElementById("tipo_movimiento");
  const grupoRetorno = document.getElementById("grupo_retorno");
  const grupoTrabajador = document.getElementById("grupo_trabajador");
  const grupoCliente = document.getElementById("grupo_cliente");
  const fechaRetorno = document.getElementById("fecha_retorno");
  const tablaMovimientosBody = document.querySelector("#tablaMovimientos tbody");

  // instancia de DataTable (única)
  let dtMovimientos = null;

  // ===== Helpers de fecha =====
  function hoyISO() {
    // normaliza a zona local (evita desfasajes con el input date)
    const tz = new Date().getTimezoneOffset() * 60000;
    return new Date(Date.now() - tz).toISOString().slice(0, 10); // YYYY-MM-DD
  }
  function aplicarMinHoy() {
    if (!fechaRetorno) return;
    const today = hoyISO();
    fechaRetorno.min = today;
    // si ya tiene algo menor a hoy (ej. al volver de otra pantalla), corrígelo
    if (fechaRetorno.value && fechaRetorno.value < today) {
      fechaRetorno.value = today;
    }
  }

  // ===== UI inicial =====
  cargarTrabajadores();
  cargarMovimientos();
  aplicarMinHoy();   // fija el min del input date

  // ✅ Refresco centralizado: historial + (si existe) tabla de productos
  function refreshInventario() {
    cargarMovimientos();
    if (typeof window.cargarProductos === "function") {
      try { window.cargarProductos(); } catch (e) {}
    }
  }

  function toggleCampos() {
    const tipo = tipoMovimiento.value;
    const conRetorno = (tipo === "con_retorno");
    if (grupoRetorno)   grupoRetorno.style.display    = conRetorno ? "block" : "none";
    if (grupoTrabajador)grupoTrabajador.style.display = conRetorno ? "block" : "none";
    if (grupoCliente)   grupoCliente.style.display    = conRetorno ? "none"  : "block";
    if (fechaRetorno) {
      fechaRetorno.required = conRetorno;
      if (conRetorno) aplicarMinHoy(); else fechaRetorno.value = "";
    }
  }

  if (tipoMovimiento) {
    tipoMovimiento.addEventListener("change", toggleCampos);
    // set estado inicial
    toggleCampos();
  }

  if (form) form.addEventListener("submit", e => {
    e.preventDefault();

    // Validación extra en cliente: si es con retorno, fecha >= hoy
    if (tipoMovimiento && tipoMovimiento.value === "con_retorno") {
      const today = hoyISO();
      if (!fechaRetorno.value) {
        return Swal.fire('Atención', 'Ingrese la fecha de retorno.', 'warning');
      }
      if (fechaRetorno.value < today) {
        return Swal.fire('Atención', 'La fecha de retorno no puede ser anterior a hoy.', 'warning');
      }
    }

    const datos = new FormData(form);
    datos.append("accion", "registrar");

    fetch("../../ajax/movimiento_ajax.php", { method: "POST", body: datos })
      .then(res => res.text())
      .then(txt => {
        let r; try { r = JSON.parse(txt); } catch (e) { throw new Error("Respuesta no válida: " + txt); }
        if (r.success) {
          Swal.fire("Éxito", "Movimiento registrado", "success");
          form.reset();
          tipoMovimiento.dispatchEvent(new Event("change"));
          refreshInventario(); // 🔥 ahora refresca todo
        } else {
          Swal.fire("Error", r.message || "Error al registrar", "error");
        }
      })
      .catch(err => Swal.fire("Error", err.message || "Error en la petición", "error"));
  });

  // Escanear código general
  const btnEsc = document.getElementById("btnEscanearQR");
  if (btnEsc) btnEsc.addEventListener("click", () => {
    const modalQR = new bootstrap.Modal(document.getElementById("modalEscanerQR"));
    modalQR.show();
    iniciarEscanerQR();
  });

  function iniciarEscanerQR() {
    const video = document.getElementById("video");
    const QrScanner = window.QrScanner;
    if (!QrScanner) { Swal.fire("Error", "Librería de QR no cargada.", "error"); return; }

    const scanner = new QrScanner(video, result => {
      const valor = (result && result.data) ? String(result.data).trim() : String(result).trim();
      const input = document.getElementById("codigo_producto");
      if (input) input.value = valor;
      scanner.stop();
      bootstrap.Modal.getInstance(document.getElementById("modalEscanerQR")).hide();
      new Audio('../../assets/sounds/beep.mp3').play().catch(()=>{});
    }, { returnDetailedScanResult: true });

    scanner.start().catch(() => {
      Swal.fire("Error", "No se pudo acceder a la cámara (revisa permisos).", "error");
    });
  }

  function cargarTrabajadores() {
    fetch("../../ajax/trabajadores_ajax.php?accion=obtener", { cache: "no-store" })
      .then(res => res.json())
      .then(data => {
        const select = document.getElementById("trabajador_id");
        if (!select) return;
        select.innerHTML = '<option value="">Seleccione...</option>';
        data.forEach(t => { select.innerHTML += `<option value="${t.id}">${t.nombre}</option>`; });
      });
  }

  // === CARGA/REFRESCO DE MOVIMIENTOS ===
  function cargarMovimientos() {
    const url = `../../ajax/movimiento_ajax.php?accion=listar&t=${Date.now()}`; // cache-buster
    fetch(url, { cache: "no-store" })
      .then(res => res.json())
      .then(data => {
        // Mapeamos los datos a filas para DataTables (columnas en el mismo orden del thead)
        const rows = data.map(m => ([
          `${m.id}`,
          `${m.producto}`,
          `${m.tipo_movimiento || m.tipo || 'salida'}`,
          `${m.trabajador_nombre || m.responsable || '-'}`,
          `${m.cantidad}`,
          `${m.motivo}`,
          `${m.fecha}`,
          `${m.fecha_retorno || '-'}`,
          `${formatearEstado(m.estado_retorno, m.fecha_retorno)}`,
          `${botonesAccion(m)}`
        ]));

        // Inicializa una sola vez y luego actualiza filas sin destruir la tabla
        if (dtMovimientos) {
          dtMovimientos.clear();
          dtMovimientos.rows.add(rows).draw(false); // false = mantiene paginación
        } else {
          dtMovimientos = $('#tablaMovimientos').DataTable({
            data: rows,
            deferRender: true,
            responsive: true,
            pageLength: 5,
            dom: 'Bfrtip',
            buttons: ['excel', 'pdf', 'print'],
            language: { url: '//cdn.datatables.net/plug-ins/1.10.24/i18n/Spanish.json' }
          });
        }

        // (Compatibilidad con manipulación directa de tbody: ya no es necesario)
        if (tablaMovimientosBody && !dtMovimientos) {
          let html = "";
          data.forEach(m => {
            html += `
              <tr>
                <td>${m.id}</td>
                <td>${m.producto}</td>
                <td>${m.tipo_movimiento || m.tipo || 'salida'}</td>
                <td>${m.trabajador_nombre || m.responsable || '-'}</td>
                <td>${m.cantidad}</td>
                <td>${m.motivo}</td>
                <td>${m.fecha}</td>
                <td>${m.fecha_retorno || '-'}</td>
                <td>${formatearEstado(m.estado_retorno, m.fecha_retorno)}</td>
                <td>${botonesAccion(m)}</td>
              </tr>`;
          });
          tablaMovimientosBody.innerHTML = html;
        }
      });
  }

  function formatearEstado(estado, fecha_retorno) {
    if (!fecha_retorno) return "sin retorno";
    if (estado === "devuelto") return "✔ Devuelto";
    if (estado === "devuelto_defectuoso") return "⚠ Devuelto Defectuoso";
    return estado || "pendiente";
  }

  // Botones SOLO cuando hay fecha_retorno (o sea, con retorno) y sigue pendiente
  function botonesAccion(m) {
    const esConRetorno = !!m.fecha_retorno; // si tiene fecha, es con retorno
    if (esConRetorno && m.estado_retorno === "pendiente") {
      return `
        <button class="btn btn-sm btn-success" onclick="marcarDevuelto(${m.id})">Devuelto</button>
        <button class="btn btn-sm btn-danger" onclick="marcarDefectuoso(${m.id})">Defectuoso</button>`;
    }
    return '<span class="text-muted">—</span>';
  }

  window.marcarDevuelto = function (id) {
    const datos = new FormData();
    datos.append("accion", "marcar_devuelto");
    datos.append("movimiento_id", id);
    fetch("../../ajax/movimiento_ajax.php", { method: "POST", body: datos })
      .then(res => res.json())
      .then(resp => {
        if (resp.success) {
          Swal.fire("Actualizado", "Producto marcado como devuelto", "success");
          refreshInventario();
        } else {
          Swal.fire("Error", resp.message, "error");
        }
      });
  };

  window.marcarDefectuoso = function (id) {
    const datos = new FormData();
    datos.append("accion", "marcar_defectuoso");
    datos.append("movimiento_id", id);
    fetch("../../ajax/movimiento_ajax.php", { method: "POST", body: datos })
      .then(res => res.json())
      .then(resp => {
        if (resp.success) {
          Swal.fire("Actualizado", "Producto marcado como defectuoso", "warning");
          refreshInventario();
        } else {
          Swal.fire("Error", resp.message, "error");
        }
      });
  };

  // Nuevo trabajador
  const btnNuevoTrabajador = document.getElementById("btnNuevoTrabajador");
  if (btnNuevoTrabajador) btnNuevoTrabajador.addEventListener("click", () => {
    const modal = new bootstrap.Modal(document.getElementById("modalTrabajador"));
    document.getElementById("formTrabajador").reset();
    modal.show();
  });

  const formTrabajador = document.getElementById("formTrabajador");
  if (formTrabajador) formTrabajador.addEventListener("submit", e => {
    e.preventDefault();
    const datos = new FormData(formTrabajador);
    datos.append("accion", "registrar");
    fetch("../../ajax/trabajadores_ajax.php", { method: "POST", body: datos })
      .then(res => res.json())
      .then(r => {
        if (r.success) {
          Swal.fire("Éxito", "Trabajador registrado", "success");
          bootstrap.Modal.getInstance(document.getElementById("modalTrabajador")).hide();
          formTrabajador.reset(); cargarTrabajadores();
        } else { Swal.fire("Error", r.message, "error"); }
      });
  });
});
